import React, { useEffect, useState, useRef } from "react";
import { useParams } from "react-router-dom";
import { api } from "../config/api";
import { motion, AnimatePresence } from "framer-motion";
import { ChevronRight, ChevronLeft, Check } from "lucide-react";

export const Form = () => {
  const [form, setForm] = useState(null);
  const [answers, setAnswers] = useState({});
  const [currentStep, setCurrentStep] = useState(0);
  const [submitted, setSubmitted] = useState(false);
  const { id } = useParams();
  const inputRef = useRef(null);

  useEffect(() => {
    const fetchForm = async () => {
      try {
        const res = await api.get(`/api/forms/${id}`);
        setForm(res.data);
      } catch (err) {
        console.error("Failed to load form", err);
      }
    };
    fetchForm();
  }, [id]);

  useEffect(() => {
    if (inputRef.current) {
      inputRef.current.focus();
    }
  }, [currentStep, form]);

  const handleInputChange = (fieldLabel, value) => {
    setAnswers({ ...answers, [fieldLabel]: value });
  };

  const handleNext = () => {
    if (currentStep < form.steps.length - 1) {
      setCurrentStep(currentStep + 1);
    } else {
      handleSubmit();
    }
  };

  const handleBack = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  };

  const handleKeyDown = (e) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      handleNext();
    }
  };

  const handleSubmit = async () => {
    try {
      await api.post("/api/responses", {
        formId: id,
        answers,
      });
      setSubmitted(true);
    } catch (err) {
      console.error(err);
      alert("Failed to submit");
    }
  };

  if (!form)
    return (
      <div className="flex justify-center items-center h-screen bg-gray-50 text-neutral-900">
        <div className="animate-pulse">Loading...</div>
      </div>
    );

  if (submitted) {
    return (
      <div className="flex flex-col justify-center items-center h-screen bg-gray-50 text-neutral-900 px-4 text-center">
        <motion.div
          initial={{ opacity: 0, scale: 0.9 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ duration: 0.5 }}
          className="max-w-md w-full"
        >
          <div className="mx-auto w-16 h-16 bg-green-500 rounded-full flex items-center justify-center mb-6 text-white">
            <Check size={32} />
          </div>
          <h1 className="text-4xl font-light mb-4">All done!</h1>
          <p className="text-xl text-neutral-500">Thanks for your time.</p>
        </motion.div>
      </div>
    );
  }

  const step = form.steps[currentStep];
  const progress = ((currentStep + 1) / form.steps.length) * 100;

  return (
    <div className="min-h-screen bg-gray-50 text-neutral-900 flex flex-col overflow-hidden font-sans selection:bg-blue-100 selection:text-blue-900">
      {/* Progress Bar */}
      <div className="fixed top-0 left-0 w-full h-1 bg-gray-200 z-50">
        <motion.div
          className="h-full bg-blue-600"
          initial={{ width: 0 }}
          animate={{ width: `${progress}%` }}
          transition={{ duration: 0.5 }}
        />
      </div>

      <div className="flex-1 flex flex-col items-center justify-center w-full max-w-3xl mx-auto px-6 relative">
        <AnimatePresence mode="wait" initial={false}>
          <motion.div
            key={currentStep}
            initial={{ opacity: 0, y: 50 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -50 }}
            transition={{ duration: 0.4, ease: "easeOut" }}
            className="w-full"
          >
            <div className="mb-4 text-blue-600 text-sm font-medium uppercase tracking-widest">
              {currentStep + 1} <span className="text-neutral-400">/</span>{" "}
              {form.steps.length}
            </div>

            <h2 className="text-3xl md:text-4xl lg:text-5xl font-light mb-8 md:mb-12 leading-tight text-neutral-900">
              {step.title}
            </h2>

            <div className="space-y-8">
              {step.fields?.map((field, index) => (
                <div key={index} className="flex flex-col group">
                  <label className="text-xl text-neutral-600 mb-4 block font-medium">
                    {field.label}{" "}
                    <span className="text-red-500 text-sm">*</span>
                  </label>

                  {field.type === "select" ? (
                    <div className="relative">
                      <select
                        ref={index === 0 ? inputRef : null}
                        className="w-full bg-transparent border-b-2 border-gray-300 py-4 text-2xl md:text-3xl focus:border-blue-600 focus:outline-none transition-colors appearance-none cursor-pointer text-neutral-900"
                        onChange={(e) =>
                          handleInputChange(field.label, e.target.value)
                        }
                        value={answers[field.label] || ""}
                        onKeyDown={handleKeyDown}
                      >
                        <option
                          value=""
                          disabled
                          className="bg-white text-gray-400"
                        >
                          Select an option...
                        </option>
                        {field.options?.map((opt, i) => (
                          <option
                            key={i}
                            value={opt}
                            className="bg-white text-neutral-900"
                          >
                            {opt}
                          </option>
                        ))}
                      </select>
                      <div className="absolute right-0 top-1/2 -translate-y-1/2 text-neutral-400 pointer-events-none">
                        ▼
                      </div>
                    </div>
                  ) : field.type === "textarea" ? (
                    <textarea
                      ref={index === 0 ? inputRef : null}
                      className="w-full bg-transparent border-b-2 border-gray-300 py-4 text-2xl md:text-3xl focus:border-blue-600 focus:outline-none transition-colors resize-none placeholder:text-gray-300 text-neutral-900"
                      rows={2}
                      onChange={(e) =>
                        handleInputChange(field.label, e.target.value)
                      }
                      value={answers[field.label] || ""}
                      onKeyDown={(e) => {
                        if (e.key === "Enter" && e.ctrlKey) handleNext();
                      }}
                      placeholder="Type your answer here..."
                    />
                  ) : (
                    <input
                      ref={index === 0 ? inputRef : null}
                      type={field.type}
                      className="w-full bg-transparent border-b-2 border-gray-300 py-4 text-2xl md:text-3xl focus:border-blue-600 focus:outline-none transition-colors placeholder:text-gray-300 text-neutral-900"
                      onChange={(e) =>
                        handleInputChange(field.label, e.target.value)
                      }
                      value={answers[field.label] || ""}
                      onKeyDown={handleKeyDown}
                      placeholder="Type your answer here..."
                    />
                  )}
                </div>
              ))}
            </div>

            <div className="mt-12 flex items-center gap-4">
              <button
                onClick={handleNext}
                className="bg-blue-600 text-white px-8 py-3 rounded text-xl font-medium hover:bg-blue-700 active:scale-95 transition-all flex items-center gap-2 shadow-lg shadow-blue-200"
              >
                {currentStep === form.steps.length - 1 ? "Submit" : "OK"}
                <Check
                  size={20}
                  className={
                    currentStep === form.steps.length - 1 ? "block" : "hidden"
                  }
                />
                <Check
                  size={20}
                  className={
                    currentStep !== form.steps.length - 1 ? "hidden" : "hidden"
                  }
                />{" "}
                {/* Dumb hack to keep width */}
              </button>
            </div>
          </motion.div>
        </AnimatePresence>
      </div>

      <div className="fixed bottom-8 right-8 flex gap-2">
        <button
          onClick={handleBack}
          disabled={currentStep === 0}
          className="p-3 rounded-md bg-blue-600 text-white hover:bg-blue-700 disabled:bg-gray-200 disabled:text-gray-400 disabled:cursor-not-allowed transition-colors shadow-lg shadow-blue-100"
        >
          <ChevronLeft size={24} />
        </button>
        <button
          onClick={handleNext}
          disabled={currentStep === form.steps.length - 1} // Logic is handled in main button, this is just for nav
          className="p-3 rounded-md bg-blue-600 text-white hover:bg-blue-700 disabled:bg-gray-200 disabled:text-gray-400 disabled:cursor-not-allowed transition-colors shadow-lg shadow-blue-100"
        >
          <ChevronRight size={24} />
        </button>
      </div>
    </div>
  );
};
